"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ECALDebugClient = void 0;
const net = require("net");
const events_1 = require("events");
const promise_socket_1 = require("promise-socket");
class ECALDebugClient extends events_1.EventEmitter {
    constructor(out) {
        super();
        this.out = out;
        this.connected = false;
        this.backlog = [];
        this.threadInspection = {};
        this.doReload = false;
        this.socket = new promise_socket_1.default(new net.Socket());
        const AsyncLock = require("async-lock");
        this.socketLock = new AsyncLock();
    }
    connect(host, port) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                this.out.log(`Connecting to: ${host}:${port}`);
                yield this.socket.connect({ port, host });
                this.connected = true;
                this.pollEvents();
            }
            catch (e) {
                this.out.error(`Could not connect to debug server: ${e}`);
            }
        });
    }
    status() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                return (yield this.sendCommand("status"));
            }
            catch (e) {
                this.out.error(`Could not query for status: ${e}`);
                return null;
            }
        });
    }
    reload() {
        this.doReload = true;
    }
    describe(tid) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                return (yield this.sendCommand("describe", [
                    String(tid),
                ]));
            }
            catch (e) {
                this.out.error(`Could not inspect thread ${tid}: ${e}`);
                return null;
            }
        });
    }
    cont(tid, type) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                yield this.sendCommand("cont", [String(tid), type]);
                delete this.threadInspection[tid];
            }
            catch (e) {
                this.out.error(`Could not continue thread ${tid}: ${e}`);
            }
        });
    }
    setBreakpoint(breakpoint) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                (yield this.sendCommand(`break ${breakpoint}`));
            }
            catch (e) {
                this.out.error(`Could not set breakpoint ${breakpoint}: ${e}`);
            }
        });
    }
    clearBreakpoints(source) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                (yield this.sendCommand("rmbreak", [source]));
            }
            catch (e) {
                this.out.error(`Could not remove breakpoints for ${source}: ${e}`);
            }
        });
    }
    shutdown() {
        return __awaiter(this, void 0, void 0, function* () {
            this.connected = false;
            yield this.socket.destroy();
        });
    }
    pollEvents() {
        return __awaiter(this, void 0, void 0, function* () {
            let nextLoop = 1000;
            try {
                const status = yield this.status();
                this.emit("status", status);
                for (const [tidString, thread] of Object.entries((status === null || status === void 0 ? void 0 : status.threads) || [])) {
                    const tid = parseInt(tidString);
                    if (thread.threadRunning === false && !this.threadInspection[tid]) {
                        let inspection = {
                            callStack: [],
                            threadRunning: false,
                        };
                        try {
                            inspection = (yield this.sendCommand("describe", [
                                String(tid),
                            ]));
                        }
                        catch (e) {
                            this.out.error(`Could not get description for ${tid}: ${e}`);
                        }
                        this.threadInspection[tid] = inspection;
                        this.emit("pauseOnBreakpoint", { tid, inspection });
                    }
                }
                if (this.doReload) {
                    this.doReload = false;
                    this.out.log(`Reloading interpreter state`);
                    try {
                        yield this.sendCommandString("@reload\r\n");
                    }
                    catch (e) {
                        this.out.error(`Could not reload the interpreter state: ${e}`);
                    }
                }
            }
            catch (e) {
                this.out.error(`Error during event loop: ${e}`);
                nextLoop = 5000;
            }
            if (this.connected) {
                setTimeout(this.pollEvents.bind(this), nextLoop);
            }
            else {
                this.out.log("Stop emitting events" + nextLoop);
            }
        });
    }
    sendCommand(cmd, args) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.connected) {
                this.backlog.push({
                    cmd,
                    args,
                });
                return null;
            }
            else if (this.backlog.length > 0) {
                const backlog = this.backlog;
                this.backlog = [];
                for (const item of backlog) {
                    yield this.sendCommand(item.cmd, item.args);
                }
            }
            return yield this.sendCommandString(`##${cmd} ${args ? args.join(" ") : ""}\r\n`);
        });
    }
    sendCommandString(cmdString) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.socketLock.acquire("socket", () => __awaiter(this, void 0, void 0, function* () {
                yield this.socket.write(cmdString, "utf8");
                let text = "";
                while (!text.endsWith("\n\n")) {
                    text += yield this.socket.read(1);
                }
                let res = {};
                try {
                    res = JSON.parse(text);
                }
                catch (e) {
                    throw new Error(`Could not parse response: ${text} - error:${e}`);
                }
                if (res === null || res === void 0 ? void 0 : res.DebuggerError) {
                    throw new Error(`Unexpected internal error for command "${cmdString}": ${res.DebuggerError}`);
                }
                if ((res === null || res === void 0 ? void 0 : res.EncodedOutput) !== undefined) {
                    res = Buffer.from(res.EncodedOutput, "base64").toString("utf8");
                }
                return res;
            }));
        });
    }
}
exports.ECALDebugClient = ECALDebugClient;
//# sourceMappingURL=ecalDebugClient.js.map