"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ECALDebugSession = void 0;
const vscode_debugadapter_1 = require("vscode-debugadapter");
const waitgroup_1 = require("@jpwilliams/waitgroup");
const ecalDebugClient_1 = require("./ecalDebugClient");
const vscode = require("vscode");
const types_1 = require("./types");
const path = require("path");
class ECALDebugSession extends vscode_debugadapter_1.LoggingDebugSession {
    constructor() {
        super("mock-debug.txt");
        this.extout = vscode.window.createOutputChannel("ECAL Debug Session");
        this.wgConfig = new waitgroup_1.WaitGroup();
        this.config = {};
        this.unconfirmedBreakpoints = [];
        this.frameVariableScopes = {};
        this.frameVariableGlobalScopes = {};
        this.bpCount = 1;
        this.sfCount = 1;
        this.vsCount = 1;
        this.bpIds = {};
        this.sfIds = {};
        this.vsIds = {};
        this.extout.appendLine("Creating Debug Session");
        this.client = new ecalDebugClient_1.ECALDebugClient(new LogChannelAdapter(this.extout));
        this.client.on("pauseOnBreakpoint", (e) => {
            this.sendEvent(new vscode_debugadapter_1.StoppedEvent("breakpoint", e.tid));
        });
        this.client.on("status", (e) => {
            var _a;
            try {
                if (this.unconfirmedBreakpoints.length > 0) {
                    for (const toConfirm of this.unconfirmedBreakpoints) {
                        for (const [breakpointString, ok] of Object.entries(e.breakpoints)) {
                            const line = parseInt(breakpointString.split(":")[1]);
                            if (ok) {
                                if (toConfirm.line === line &&
                                    ((_a = toConfirm.source) === null || _a === void 0 ? void 0 : _a.name) === breakpointString) {
                                    toConfirm.verified = true;
                                    this.sendEvent(new vscode_debugadapter_1.BreakpointEvent("changed", toConfirm));
                                }
                            }
                        }
                    }
                    this.unconfirmedBreakpoints = [];
                }
            }
            catch (e) {
                console.error(e);
            }
        });
        this.setDebuggerLinesStartAt1(true);
        this.setDebuggerColumnsStartAt1(true);
        this.wgConfig.add(1);
    }
    initializeRequest(response) {
        response.body = response.body || {};
        response.body.supportsConfigurationDoneRequest = true;
        response.body.supportsBreakpointLocationsRequest = true;
        response.body.supportsStepInTargetsRequest = true;
        this.sendResponse(response);
        this.sendEvent(new vscode_debugadapter_1.InitializedEvent());
    }
    configurationDoneRequest(response, args) {
        super.configurationDoneRequest(response, args);
        this.wgConfig.done();
    }
    launchRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            this.config = args;
            vscode_debugadapter_1.logger.setup(args.trace ? vscode_debugadapter_1.Logger.LogLevel.Verbose : vscode_debugadapter_1.Logger.LogLevel.Error, false);
            yield this.wgConfig.wait();
            this.extout.appendLine(`Configuration loaded: ${JSON.stringify(args)}`);
            yield this.client.connect(args.host, args.port);
            if (args.executeOnEntry) {
                this.client.reload();
            }
            this.sendResponse(response);
        });
    }
    setBreakPointsRequest(response, args) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            let breakpoints = [];
            if (((_a = args.source.path) === null || _a === void 0 ? void 0 : _a.indexOf(this.config.dir)) === 0) {
                const sourcePath = args.source.path.slice(this.config.dir.length + 1);
                yield this.client.clearBreakpoints(sourcePath);
                for (const sbp of args.breakpoints || []) {
                    yield this.client.setBreakpoint(`${sourcePath}:${sbp.line}`);
                }
                const status = yield this.client.status();
                if (status) {
                    breakpoints = (args.lines || []).map((line) => {
                        const breakpointString = `${sourcePath}:${line}`;
                        const bp = new vscode_debugadapter_1.Breakpoint(status.breakpoints[breakpointString], line, undefined, new vscode_debugadapter_1.Source(breakpointString, args.source.path));
                        bp.id = this.getBreakPointId(breakpointString);
                        return bp;
                    });
                }
                else {
                    breakpoints = (args.breakpoints || []).map((sbp) => {
                        const breakpointString = `${sourcePath}:${sbp.line}`;
                        const bp = new vscode_debugadapter_1.Breakpoint(false, sbp.line, undefined, new vscode_debugadapter_1.Source(breakpointString, args.source.path));
                        bp.id = this.getBreakPointId(breakpointString);
                        return bp;
                    });
                    this.unconfirmedBreakpoints = breakpoints;
                }
            }
            response.body = {
                breakpoints,
            };
            this.sendResponse(response);
        });
    }
    breakpointLocationsRequest(response, args) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            let breakpoints = [];
            if (((_a = args.source.path) === null || _a === void 0 ? void 0 : _a.indexOf(this.config.dir)) === 0) {
                const sourcePath = args.source.path.slice(this.config.dir.length + 1);
                const status = yield this.client.status();
                if (status) {
                    for (const [breakpointString, v] of Object.entries(status.breakpoints)) {
                        if (v) {
                            const line = parseInt(breakpointString.split(":")[1]);
                            if (`${sourcePath}:${line}` === breakpointString) {
                                breakpoints.push({
                                    line,
                                });
                            }
                        }
                    }
                }
            }
            response.body = {
                breakpoints,
            };
            this.sendResponse(response);
        });
    }
    threadsRequest(response) {
        return __awaiter(this, void 0, void 0, function* () {
            const status = yield this.client.status();
            const threads = [];
            if (status) {
                for (const tid of Object.keys(status.threads)) {
                    threads.push(new vscode_debugadapter_1.Thread(parseInt(tid), `Thread ${tid}`));
                }
            }
            else {
                threads.push(new vscode_debugadapter_1.Thread(1, "Thread 1"));
            }
            response.body = {
                threads,
            };
            this.sendResponse(response);
        });
    }
    stackTraceRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            const stackFrames = [];
            const status = yield this.client.status();
            const threadStatus = status === null || status === void 0 ? void 0 : status.threads[String(args.threadId)];
            if ((threadStatus === null || threadStatus === void 0 ? void 0 : threadStatus.threadRunning) === false) {
                const ins = yield this.client.describe(args.threadId);
                if (ins) {
                    for (const [i, sf] of ins.callStack.entries()) {
                        const sfNode = ins.callStackNode[i];
                        const frameId = this.getStackFrameId(args.threadId, sf, i);
                        const breakpointString = `${sfNode.source}:${sfNode.line}`;
                        stackFrames.unshift(new vscode_debugadapter_1.StackFrame(frameId, sf, new vscode_debugadapter_1.Source(breakpointString, path.join(this.config.dir, sfNode.source)), sfNode.line));
                        this.frameVariableScopes[frameId] = ins.callStackVsSnapshot[i];
                        this.frameVariableGlobalScopes[frameId] = ins.callStackVsSnapshotGlobal[i];
                    }
                    const frameId = this.getStackFrameId(args.threadId, ins.code, ins.callStack.length);
                    const breakpointString = `${ins.node.source}:${ins.node.line}`;
                    stackFrames.unshift(new vscode_debugadapter_1.StackFrame(frameId, ins.code, new vscode_debugadapter_1.Source(breakpointString, path.join(this.config.dir, ins.node.source)), ins.node.line));
                    this.frameVariableScopes[frameId] = ins.vs;
                    this.frameVariableGlobalScopes[frameId] = ins.vsGlobal;
                }
            }
            response.body = {
                stackFrames,
            };
            this.sendResponse(response);
        });
    }
    scopesRequest(response, args) {
        response.body = {
            scopes: [
                new vscode_debugadapter_1.Scope("Local", this.getVariableScopeId(args.frameId, "local")),
                new vscode_debugadapter_1.Scope("Global", this.getVariableScopeId(args.frameId, "global")),
            ],
        };
        this.sendResponse(response);
    }
    variablesRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            let vs = {};
            let variables = [];
            const [frameId, scopeType] = this.getScopeLookupInfo(args.variablesReference);
            if (scopeType === "local") {
                vs = this.frameVariableScopes[frameId];
            }
            else if (scopeType === "global") {
                vs = this.frameVariableGlobalScopes[frameId];
            }
            if (vs) {
                for (const [name, val] of Object.entries(vs)) {
                    let valString;
                    try {
                        valString = JSON.stringify(val);
                    }
                    catch (e) {
                        valString = String(val);
                    }
                    variables.push(new vscode_debugadapter_1.Variable(name, valString));
                }
            }
            response.body = {
                variables,
            };
            this.sendResponse(response);
        });
    }
    continueRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.client.cont(args.threadId, types_1.ContType.Resume);
            response.body = {
                allThreadsContinued: false,
            };
            this.sendResponse(response);
        });
    }
    nextRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.client.cont(args.threadId, types_1.ContType.StepOver);
            this.sendResponse(response);
        });
    }
    stepInRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.client.cont(args.threadId, types_1.ContType.StepIn);
            this.sendResponse(response);
        });
    }
    stepOutRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.client.cont(args.threadId, types_1.ContType.StepOut);
            this.sendResponse(response);
        });
    }
    evaluateRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            let result;
            try {
                result = yield this.client.sendCommandString(`${args.expression}\r\n`);
                if (typeof result !== "string") {
                    result = JSON.stringify(result, null, "  ");
                }
            }
            catch (e) {
                result = String(e);
            }
            response.body = {
                result,
                variablesReference: 0,
            };
            this.sendResponse(response);
        });
    }
    shutdown() {
        var _a;
        (_a = this.client) === null || _a === void 0 ? void 0 : _a.shutdown().then(() => {
            this.extout.appendLine("Debug Session has finished");
        }).catch((e) => {
            this.extout.appendLine(`Debug Session has finished with an error: ${e}`);
        });
    }
    getBreakPointId(breakpointString) {
        let id = this.bpIds[breakpointString];
        if (!id) {
            id = this.bpCount++;
            this.bpIds[breakpointString] = id;
        }
        return id;
    }
    getStackFrameId(threadId, frameString, frameIndex) {
        const storageString = `${threadId}###${frameString}###${frameIndex}`;
        let id = this.sfIds[storageString];
        if (!id) {
            id = this.sfCount++;
            this.sfIds[storageString] = id;
        }
        return id;
    }
    getVariableScopeId(frameId, scopeType) {
        const storageString = `${frameId}###${scopeType}`;
        let id = this.vsIds[storageString];
        if (!id) {
            id = this.vsCount++;
            this.vsIds[storageString] = id;
        }
        return id;
    }
    getScopeLookupInfo(vsId) {
        for (const [k, v] of Object.entries(this.vsIds)) {
            if (v === vsId) {
                const s = k.split("###");
                return [parseInt(s[0]), s[1]];
            }
        }
        return [-1, ""];
    }
}
exports.ECALDebugSession = ECALDebugSession;
class LogChannelAdapter {
    constructor(out) {
        this.out = out;
    }
    log(value) {
        this.out.appendLine(value);
    }
    error(value) {
        this.out.appendLine(`Error: ${value}`);
        setTimeout(() => {
            this.out.show(true);
        }, 500);
    }
}
//# sourceMappingURL=ecalDebugAdapter.js.map