"use strict";
exports.__esModule = true;
exports.WaitGroup = void 0;
var events_1 = require("events");
/**
 * A WaitGroup waits for a collection of actions to finish.
 * The main goroutine calls `add` to set the number of actions to wait for.
 * Then each of the actions runs and calls `done` when finished.
 * At the same time, `wait` can be used to return a promise that resolves when all actions have finished.
 *
 * The class doesn't implement the race-condition requirements that the Golang package has due to the
 * way Node functions, meaning `add`, `done` and `wait` can be called at any time, in any order.
 */
var WaitGroup = /** @class */ (function () {
    function WaitGroup() {
        this._current = 0;
        this._emitter = new events_1.EventEmitter();
    }
    /**
     * Adds a delta, which may be negative, to the WaitGroup counter.
     * If the counter becomes zero, all promises returned from `wait` are resolved.
     * If the counter goes negative, an error is thrown.
     */
    WaitGroup.prototype.add = function (delta) {
        if (delta === void 0) { delta = 1; }
        this._current += delta;
        if (this._current < 0)
            throw new Error('Negative WaitGroup counter');
        if (this._current === 0)
            this._emitter.emit('done');
    };
    /**
     * Decrements the WaitGroup counter by one.
     */
    WaitGroup.prototype.done = function () {
        this.add(-1);
    };
    /**
     * Returns a promise that resolves when the WaitGroup counter is zero.
     * If the counter is zero when the method is called, it's resolved immediately.
     */
    WaitGroup.prototype.wait = function () {
        var _this = this;
        return new Promise(function (resolve) {
            if (_this._current === 0)
                return resolve();
            _this._emitter.once('done', function () { return resolve(); });
        });
    };
    return WaitGroup;
}());
exports.WaitGroup = WaitGroup;
