"use strict";
/// <reference types="node" />
Object.defineProperty(exports, "__esModule", { value: true });
exports.PromiseReadable = void 0;
require("core-js/modules/es.symbol.async-iterator");
class PromiseReadable {
    constructor(stream) {
        this.stream = stream;
        this._isPromiseReadable = true;
        this.errorHandler = (err) => {
            this._errored = err;
        };
        stream.on("error", this.errorHandler);
    }
    static [Symbol.hasInstance](instance) {
        return instance._isPromiseReadable;
    }
    read(size) {
        const stream = this.stream;
        return new Promise((resolve, reject) => {
            if (this._errored) {
                const err = this._errored;
                this._errored = undefined;
                return reject(err);
            }
            if (!stream.readable || stream.closed || stream.destroyed) {
                return resolve();
            }
            const readableHandler = () => {
                const chunk = stream.read(size);
                if (chunk !== null) {
                    removeListeners();
                    resolve(chunk);
                }
            };
            const closeHandler = () => {
                removeListeners();
                resolve();
            };
            const endHandler = () => {
                removeListeners();
                resolve();
            };
            const errorHandler = (err) => {
                this._errored = undefined;
                removeListeners();
                reject(err);
            };
            const removeListeners = () => {
                stream.removeListener("close", closeHandler);
                stream.removeListener("error", errorHandler);
                stream.removeListener("end", endHandler);
                stream.removeListener("readable", readableHandler);
            };
            stream.on("close", closeHandler);
            stream.on("end", endHandler);
            stream.on("error", errorHandler);
            stream.on("readable", readableHandler);
            readableHandler();
        });
    }
    readAll() {
        const stream = this.stream;
        const bufferArray = [];
        let content = "";
        return new Promise((resolve, reject) => {
            if (this._errored) {
                const err = this._errored;
                this._errored = undefined;
                return reject(err);
            }
            if (!stream.readable || stream.closed || stream.destroyed) {
                return resolve();
            }
            const dataHandler = (chunk) => {
                if (typeof chunk === "string") {
                    content += chunk;
                }
                else {
                    bufferArray.push(chunk);
                }
            };
            const closeHandler = () => {
                removeListeners();
                resolve();
            };
            const endHandler = () => {
                removeListeners();
                if (bufferArray.length) {
                    resolve(Buffer.concat(bufferArray));
                }
                else {
                    resolve(content);
                }
            };
            const errorHandler = (err) => {
                this._errored = undefined;
                removeListeners();
                reject(err);
            };
            const removeListeners = () => {
                stream.removeListener("close", closeHandler);
                stream.removeListener("data", dataHandler);
                stream.removeListener("error", errorHandler);
                stream.removeListener("end", endHandler);
            };
            stream.on("close", closeHandler);
            stream.on("data", dataHandler);
            stream.on("end", endHandler);
            stream.on("error", errorHandler);
            stream.resume();
        });
    }
    setEncoding(encoding) {
        this.stream.setEncoding(encoding);
        return this;
    }
    once(event) {
        const stream = this.stream;
        return new Promise((resolve, reject) => {
            if (this._errored) {
                const err = this._errored;
                this._errored = undefined;
                return reject(err);
            }
            if (stream.closed) {
                if (event === "close") {
                    return resolve();
                }
                else {
                    return reject(new Error(`once ${event} after close`));
                }
            }
            else if (stream.destroyed) {
                if (event === "close" || event === "end") {
                    return resolve();
                }
                else {
                    return reject(new Error(`once ${event} after destroy`));
                }
            }
            const closeHandler = () => {
                removeListeners();
                resolve();
            };
            const eventHandler = event !== "close" && event !== "end" && event !== "error"
                ? (argument) => {
                    removeListeners();
                    resolve(argument);
                }
                : undefined;
            const endHandler = event !== "close"
                ? () => {
                    removeListeners();
                    resolve();
                }
                : undefined;
            const errorHandler = (err) => {
                this._errored = undefined;
                removeListeners();
                reject(err);
            };
            const removeListeners = () => {
                if (eventHandler) {
                    stream.removeListener(event, eventHandler);
                }
                stream.removeListener("error", errorHandler);
                if (endHandler) {
                    stream.removeListener("end", endHandler);
                }
                stream.removeListener("error", errorHandler);
            };
            if (eventHandler) {
                stream.on(event, eventHandler);
            }
            stream.on("close", closeHandler);
            if (endHandler) {
                stream.on("end", endHandler);
            }
            stream.on("error", errorHandler);
        });
    }
    iterate(size) {
        const promiseReadable = this;
        let wasEof = false;
        return {
            [Symbol.asyncIterator]() {
                return this;
            },
            async next() {
                if (wasEof) {
                    return { value: "", done: true };
                }
                else {
                    const value = await promiseReadable.read(size);
                    if (value === undefined) {
                        wasEof = true;
                        return { value: "", done: true };
                    }
                    else {
                        return { value, done: false };
                    }
                }
            },
        };
    }
    [Symbol.asyncIterator]() {
        return this.iterate();
    }
    destroy() {
        if (this.stream) {
            this.stream.removeListener("error", this.errorHandler);
            if (typeof this.stream.destroy === "function") {
                this.stream.destroy();
            }
        }
    }
}
exports.PromiseReadable = PromiseReadable;
exports.default = PromiseReadable;
//# sourceMappingURL=promise-readable.js.map